<#
.SYNOPSIS
    Enable remote access for given users
.DESCRIPTION
    Enable remote access via Remote Connector Service for given users. Prompts for
    root certificate password if not given as parameter. This script does the 
    following:

    - Enables SwyxWare username/password login for the user if not already enabled
    - Create Remote Connector client certificate if not already present for the user
    - Sends welcome email if either certificate has been created or login has been 
      enabled.
.PARAMETER UserEntry
    User to enable remote access for.
.PARAMETER RootCertificatePassword
    Remote Connector Root Certificate password as defined during root certificate creation.
    Enable-IpPbxUserRemoteAccess will prompt for the password when not specified.
.PARAMETER SkipSendingWelcomeEmail 
    Skip sending welcome emails
.PARAMETER OutputAccountData
    Outputs username and password to allow delivering them to users in other ways than 
    sending welcome email
.EXAMPLE
    Get-IpPbxUser | Enable-IpPbxUserRemoteAccess 
    
    Enable remote access for all users 
.EXAMPLE
    Get-IpPbxUser -Username "Alice" | Enable-IpPbxUserRemoteAccess -SkipSendingWelcomeEmail

    Enable remote access for user with name "Alice", but do not send welcome email
.EXAMPLE
    Get-IpPbxUser | Enable-IpPbxUserRemoteAccess -RootCertificatePassword $Password
    
    Enable remote access for all users using password stored in $Password as SecureString as root cert password
.EXAMPLE
    Get-IpPbxGroupMember -GroupName "RemoteAccessSwyxWareUser" | Enable-IpPbxUserRemoteAccess
    
    Enable remote access for all members of SwyxWare Group "RemoteAccessSwyxWareUser"
.EXAMPLE
    Get-IpPbxUser | Enable-IpPbxUserRemoteAccess -OutputAccountData | Export-Csv -NoTypeInformation -Path .\Users.csv
    
    Enable remote access for all users and save account data (username, generated password and email) into users.csv
 .INPUTS
    User objects.
.OUTPUTS
    None
#>
[CmdletBinding(SupportsShouldProcess=$True)]
param
(
   [Parameter(Mandatory=$True, ValueFromPipeline=$True)]
   $UserEntry,
   
   [Parameter(Mandatory=$True)][SecureString]
   $RootCertificatePassword = (Read-Host -Prompt "Root Certificate Password" -AsSecureString),

   [switch]$SkipSendingWelcomeEmail = $False,
   
   [switch]$OutputAccountData = $false
)

BEGIN {
    Set-StrictMode -Version Latest

    Add-Type -AssemblyName System.Web

    if (!(Get-IpPbxServerCertificate)) {
        throw "Found no server certificate. Remote Access not configured properly on '$($LibManager.WSBaseUrl)'. Cannot continue."
    }

    $RemoteConnectorConfig = Get-IpPbxRemoteConnectorConfig
    if (!$RemoteConnectorConfig.RemoteAccessEnabled) {
        Write-Warning "Remote Access is not enabled on $($LibManager.WSBaseUrl)"
    }

    function New-AccountData
    {
        param($User, $GeneratedPassword)

        new-object "PSObject" -Property ([ordered] @{ 
            Username = $User.Name
            Password = $GeneratedPassword
            EmailAddress = if ($User.UserEmailAddressesEntryCollection.Count -gt 0) { $User.UserEmailAddressesEntryCollection[0].Address } 
        })
    }
}

PROCESS {
    Write-Verbose -Message "Enable Remote access for User $($UserEntry.Name)"
    $CertificateThumbprint = Get-IpPbxCertificateThumbprint -UserEntry $UserEntry
    $SendWelcomeMail = (!$UserEntry.IpPbxLoginAllowed) -or ($null -eq $CertificateThumbprint)

    if (!$UserEntry.IpPbxLoginAllowed -and $PSCmdlet.ShouldProcess("User $($UserEntry.Name)", "Enable SwyxWare Login")) {
        $IpPbxPassword = [System.Web.Security.Membership]::GeneratePassword(8,2)
        Set-IpPbxUserLogin -UserEntry $UserEntry -Password $IpPbxPassword -EnableLogin -PasswordExpired 
        $UserEntry.Update($True)
        if ($OutputAccountData) {
            New-AccountData -User $UserEntry -GeneratedPassword $IpPbxPassword
        }
    }

    if (($null -eq $CertificateThumbprint) -and $PSCmdlet.ShouldProcess("User $($UserEntry.Name)", "Create RC client certificate")) {
        New-IpPbxClientCertificate -UserEntry $UserEntry -RootPassword $RootCertificatePassword -Confirm:$false | Out-Null
    }

    if ($SendWelcomeMail -and !$SkipSendingWelcomeEmail -and $PSCmdlet.ShouldProcess("User $($UserEntry.Name)", "Send Welcome Email")) {
        Send-IpPbxUserWelcomeMail -UserEntry $UserEntry 
    }
}